<?php
namespace WooBase\Options;
use WooBase\Utility\Encryption;

// Prevent direct access to this file
defined("ABSPATH") || exit();

/**
 * Class GlobalOptions
 *
 * Registers global options
 */
class GlobalOptions
{
  /**
   * GlobalOptions constructor.
   */
  public function __construct()
  {
    add_action("admin_init", ["WooBase\Options\GlobalOptions", "create_global_option"]);
    add_action("rest_api_init", ["WooBase\Options\GlobalOptions", "create_global_option"]);
  }

  /**
   * Creates global option
   *
   * @return Array
   * @since 3.2.13
   */
  public static function create_global_option()
  {
    $args = [
      "type" => "object",
      "sanitize_callback" => ["WooBase\Options\GlobalOptions", "sanitize_global_settings"],
      "default" => [],
      "capability" => "manage_options",
      "show_in_rest" => [
        "schema" => [
          "type" => "object",
          "properties" => [
            "license_key" => [
              "type" => "string",
            ],
            "instance_id" => [
              "type" => "string",
            ],
            "menu_name" => [
              "type" => "string",
            ],
            "load_on_page" => [
              "type" => "array",
            ],
            "logo" => [
              "type" => "string",
            ],
            "notification_sound" => [
              "type" => "boolean",
            ],
            "menu_links" => [
              "type" => "array",
            ],
          ],
        ],
      ],
    ];
    register_setting("woobase", "woobase_settings", $args);
  }

  public static function sanitize_global_settings($value)
  {
    error_log(json_encode($value));

    $sanitized_value = [];
    $options = get_site_option("woobase_settings", false);
    $options = !$options ? [] : $options;

    if (isset($value["license_key"])) {
      $sanitized_value["license_key"] = sanitize_text_field($value["license_key"]);
    }

    if (isset($value["instance_id"])) {
      $sanitized_value["instance_id"] = sanitize_text_field($value["instance_id"]);
    }

    if (isset($value["menu_name"])) {
      $sanitized_value["menu_name"] = sanitize_text_field($value["menu_name"]);
    }

    if (isset($value["load_on_page"]) && is_array($value["load_on_page"])) {
      $cleaned = [];
      foreach ($value["load_on_page"] as $item) {
        $cleaned[] = sanitize_text_field($item);
      }
      $sanitized_value["load_on_page"] = $cleaned;
    }

    if (isset($value["logo"])) {
      $sanitized_value["logo"] = sanitize_text_field($value["logo"]);
    }

    if (isset($value["menu_links"]) && is_array($value["menu_links"])) {
      $formattedMenuLinks = [];
      foreach ($value["menu_links"] as $link) {
        if (is_array($link)) {
          $sanitized_link = [
            "name" => isset($link["name"]) ? sanitize_text_field($link["name"]) : "",
            "category" => isset($link["category"]) ? sanitize_text_field($link["category"]) : "",
            "icon" => isset($link["icon"]) ? sanitize_text_field($link["icon"]) : "",
            "url" => isset($link["url"]) ? sanitize_text_field($link["url"]) : "",
            "external" => isset($link["external"]) ? (bool) $link["external"] : "",
          ];
          $formattedMenuLinks[] = $sanitized_link;
        }
      }
      $sanitized_value["menu_links"] = $formattedMenuLinks;
    }

    if (isset($value["notification_sound"])) {
      $sanitized_value["notification_sound"] = (bool) $value["notification_sound"];
    }

    return array_merge($options, $sanitized_value);
  }
}
